package tictactoe;

public class Game implements GameInterface {

  private final int XCHAR = 'X';
  private final int OCHAR = 'O';
  private final int DRAWCHAR = 'D';
  private final int EMPTYCHAR = ' ';

  private int boardSize;
  private UserInterface userX = null;
  private UserInterface userO = null;
  private char winner;
  private char[][] board;
  private UserInterface playerToMove;


  public Game(int boardSize) {
    this.boardSize = boardSize;
    board = new char[boardSize][boardSize];
    for (int i=0;i<boardSize;i++)
      for (int j=0;j<boardSize;j++)
        board[i][j] = ' ';
  }

  public int getBoardSize() {
    return boardSize;
  }


  /*
  if the move is valid:
      update the board
      display board for both users
      if game not over
         prompt next user to make a move
      otherwise
         notify both users
   if the move is invalid:
      display the board for both users
      prompt same user for move

  */
  public void move(int row, int column, UserInterface user){
     if (user != playerToMove)
       return;
     if (board[row][column] != ' ') {
       user.promptForMove();
       return;
     }
     if (user == userX) {
       board[row][column] = 'X';
       playerToMove = userO;
     }
     else {
       board[row][column] = 'O';
       playerToMove = userX;
     }
     userX.displayBoard(board);
     userO.displayBoard(board);
     if (checkGameOver()) {
       userX.gameOver(winner);
       userO.gameOver(winner);
       playerToMove = null;
     }
     else
       playerToMove.promptForMove();
  }

  private boolean checkGameOver() {
    setWinner();
    if (winner == EMPTYCHAR)
      return false;
    return true;
  }

  public void join(UserInterface user) {
     if (userX == null) {
       userX = user;
       playerToMove = userX;
     }
     else if (userO == null)
       userO = user;
  }

  public void startGame() {
    System.out.println("userX is "+userX);
      userX.displayBoard(board);
      userO.displayBoard(board);
      userX.promptForMove();
  }

  private void setWinner() {
    if (checkWinner(XCHAR))
      winner = XCHAR;
    else if (checkWinner(OCHAR))
      winner = OCHAR;
    else if (countMoves() == boardSize * boardSize)
      winner = DRAWCHAR;
    else
      winner = EMPTYCHAR;
  }

  private int countMoves() {
    int count = 0;
    for (int i = 0; i < boardSize; i++)
      for (int j = 0; j < boardSize; j++)
        if (board[i][j] != EMPTYCHAR)
          count++;
    return count;
  }

  private boolean checkWinner(int c) {
    if (checkMainDiag(c))
      return true;
    if (checkOtherDiag(c))
      return true;
    for (int i = 0; i < boardSize; i++) {
      if (checkWinnerRow(i, c))
        return true;
      if (checkWinnerCol(i, c))
        return true;
    }
    return false;
  }

  private boolean checkMainDiag(int c) {
    for (int i = 0; i < boardSize; i++)
      if (board[i][i] != c)
        return false;
    return true;
  }

  private boolean checkOtherDiag(int c) {
    for (int i = 0; i < boardSize; i++)
      if (board[i][boardSize - i - 1] != c)
        return false;
    return true;
  }

  private boolean checkWinnerRow(int row, int c) {
    for (int i = 0; i < boardSize; i++)
      if (board[row][i] != c)
        return false;
    return true;
  }

  private boolean checkWinnerCol(int col, int c) {
    for (int i = 0; i < boardSize; i++)
      if (board[i][col] != c)
        return false;
    return true;
  }




  // return a string that can display the board nicely
// this is a convenience for a text-based interface and for debugging
  public String toString() {
    String s;
    s = "boardsize is " + boardSize + " by " + boardSize + "\n";
    if ( (winner == 'X') || (winner == 'O'))
      s = s + winner + " won!\n";
    else if (winner == 'D')
      s = s + "game is a draw\n";
    else if (playerToMove == null)
      s = s + "game not started yet\n";
//    else
//      s = s + "next player is " + playerToMove.getUserId() + "\n";

    for (int i = 0; i < boardSize; i++) {
      for (int j = 0; j < boardSize; j++) {
        s = s + " " + board[i][j] + " ";
        if (j < boardSize - 1)
          s = s + "|";
      }
      s = s + "\n";
      if (i < boardSize - 1) {
        for (int j = 0; j < boardSize; j++) {
          s = s + "---";
          if (j < boardSize - 1)
            s = s + "|";
        }
        s = s + "\n";
      }
    }
    return s;
  }

}
